import * as THREE from 'three';
import { OrbitControls } from 'three/examples/jsm/controls/OrbitControls.js';
import { Lensflare, LensflareElement } from 'three/examples/jsm/objects/Lensflare.js';
import { PlanetTerrain } from './planet/terrain.js';
import { PlanetOcean } from './planet/ocean.js';
import { PlanetAtmosphere } from './planet/atmosphere.js';
import { PlanetClouds } from './planet/clouds.js';
import { PlanetAurora } from './planet/aurora.js';
import { PlanetVegetation } from './planet/vegetation.js';
import { PlanetRings } from './skybox/rings.js';
import { Stars } from './skybox/stars.js';
import { Celestials } from './skybox/celestials.js';
import { SpaceDust } from './skybox/spacedust.js';
import { PlayerController } from './player/controller.js';
import { getBiomeData } from './planet/biomes.js';
import './style.css';

// Config
const PLANET_RADIUS = 2000;

// Scene
const scene = new THREE.Scene();

// Helper for Lensflare Texture
function createFlareTexture() {
    const canvas = document.createElement('canvas');
    canvas.width = 512;
    canvas.height = 512;
    const ctx = canvas.getContext('2d');
    
    const gradient = ctx.createRadialGradient(256, 256, 0, 256, 256, 256);
    gradient.addColorStop(0, 'rgba(255, 255, 255, 1)');
    gradient.addColorStop(0.2, 'rgba(255, 255, 224, 0.6)');
    gradient.addColorStop(0.5, 'rgba(255, 200, 160, 0.2)');
    gradient.addColorStop(1, 'rgba(0, 0, 0, 0)');
    
    ctx.fillStyle = gradient;
    ctx.fillRect(0, 0, 512, 512);
    
    return new THREE.CanvasTexture(canvas);
}

// Renderer
const renderer = new THREE.WebGLRenderer({ antialias: true });
renderer.setSize(window.innerWidth, window.innerHeight);
renderer.setPixelRatio(Math.min(window.devicePixelRatio, 2)); // Cap for performance
renderer.shadowMap.enabled = true;
renderer.shadowMap.type = THREE.PCFSoftShadowMap;
document.body.appendChild(renderer.domElement);

// Camera - near plane increased to reduce z-fighting (was 0.1)
// With planet radius 2000, we don't need sub-unit precision
const camera = new THREE.PerspectiveCamera(60, window.innerWidth / window.innerHeight, 10, 50000);
camera.position.set(0, 0, PLANET_RADIUS * 3);

// Controls
const orbitControls = new OrbitControls(camera, renderer.domElement);
orbitControls.enableDamping = true;
orbitControls.minDistance = PLANET_RADIUS + 10;
orbitControls.maxDistance = PLANET_RADIUS * 10;

// Lighting
const sunDir = new THREE.Vector3(1, 0.5, 1).normalize();
const sunLight = new THREE.DirectionalLight(0xffffff, 2.0);
sunLight.position.copy(sunDir).multiplyScalar(PLANET_RADIUS * 5);
sunLight.castShadow = true;
sunLight.shadow.mapSize.width = 2048;
sunLight.shadow.mapSize.height = 2048;
sunLight.shadow.camera.near = 100;
sunLight.shadow.camera.far = PLANET_RADIUS * 10;
sunLight.shadow.camera.left = -PLANET_RADIUS * 2;
sunLight.shadow.camera.right = PLANET_RADIUS * 2;
sunLight.shadow.camera.top = PLANET_RADIUS * 2;
sunLight.shadow.camera.bottom = -PLANET_RADIUS * 2;
sunLight.shadow.bias = -0.0001;
scene.add(sunLight);

const textureFlare = createFlareTexture();
const lensflare = new Lensflare();
lensflare.addElement(new LensflareElement(textureFlare, 700, 0, sunLight.color));
lensflare.addElement(new LensflareElement(textureFlare, 60, 0.6));
lensflare.addElement(new LensflareElement(textureFlare, 70, 0.7));
lensflare.addElement(new LensflareElement(textureFlare, 120, 0.9));
sunLight.add(lensflare);

const ambientLight = new THREE.AmbientLight(0x222222);
scene.add(ambientLight);

// Planet Components
console.log("Generating Planet...");
const terrain = new PlanetTerrain(PLANET_RADIUS);
scene.add(terrain.mesh);

const vegetation = new PlanetVegetation(PLANET_RADIUS, terrain);
scene.add(vegetation.group);

const ocean = new PlanetOcean(PLANET_RADIUS, terrain);
scene.add(ocean.mesh);

const atmosphere = new PlanetAtmosphere(PLANET_RADIUS);
scene.add(atmosphere.mesh);

const clouds = new PlanetClouds(PLANET_RADIUS);
scene.add(clouds.mesh);

const aurora = new PlanetAurora(PLANET_RADIUS);
scene.add(aurora.mesh);

const rings = new PlanetRings(PLANET_RADIUS);
scene.add(rings.mesh);

// Skybox
const stars = new Stars(10000, PLANET_RADIUS * 10);
scene.add(stars.mesh);

const celestials = new Celestials(sunDir);
scene.add(celestials.group);

const spaceDust = new SpaceDust(1000, PLANET_RADIUS * 2, PLANET_RADIUS * 5);
scene.add(spaceDust.mesh);

// Player
const player = new PlayerController(camera, renderer.domElement);

// State
let mode = 'ORBIT'; // ORBIT, PLAYER
let lastTime = performance.now();
let frameCount = 0;
let lastFpsTime = 0;

// Input
window.addEventListener('keydown', (e) => {
    if (e.code === 'KeyP') {
        toggleMode();
    }
});

function toggleMode() {
    if (mode === 'ORBIT') {
        mode = 'PLAYER';
        orbitControls.enabled = false;
        
        // Position player above surface at current camera angle?
        // Or just drop them at current camera position?
        // If camera is far, drop them closer.
        const dist = camera.position.length();
        if (dist > PLANET_RADIUS + 500) {
            // Teleport to surface below camera
            const dir = camera.position.clone().normalize();
            player.position.copy(dir.multiplyScalar(PLANET_RADIUS + 100));
        } else {
            player.position.copy(camera.position);
        }
        player.velocity.set(0,0,0);
        
        player.lock();
    } else {
        mode = 'ORBIT';
        player.unlock();
        orbitControls.enabled = true;
        
        // Reset camera up to avoid roll issues?
        // OrbitControls usually handles it.
        camera.up.set(0, 1, 0);
    }
}

// Resize
window.addEventListener('resize', () => {
    camera.aspect = window.innerWidth / window.innerHeight;
    camera.updateProjectionMatrix();
    renderer.setSize(window.innerWidth, window.innerHeight);
});

// Loop
function animate() {
    requestAnimationFrame(animate);
    
    const time = performance.now();
    const delta = Math.min((time - lastTime) / 1000, 0.1); // Cap delta
    lastTime = time;
    
    // FPS
    frameCount++;
    if (time - lastFpsTime > 1000) {
        frameCount = 0;
        lastFpsTime = time;
    }
    
    const sec = time / 1000;
    
    // Updates
    if (mode === 'ORBIT') {
        orbitControls.update();
        
        // Auto-rotate light for day/night?
        // Or rotate planet?
        // Let's rotate light slowly
        const lightSpeed = 0.05;
        sunDir.x = Math.cos(sec * lightSpeed);
        sunDir.z = Math.sin(sec * lightSpeed);
        sunDir.normalize();
        sunLight.position.copy(sunDir).multiplyScalar(PLANET_RADIUS * 5);
        
    } else {
        player.update(delta, terrain);
        
        // In player mode, sun also moves
        const lightSpeed = 0.05;
        sunDir.x = Math.cos(sec * lightSpeed);
        sunDir.z = Math.sin(sec * lightSpeed);
        sunDir.normalize();
        sunLight.position.copy(sunDir).multiplyScalar(PLANET_RADIUS * 5);
        
        // Update Biome UI
        // Calculate biome at player pos
        const pPos = player.position;
        const height = pPos.length() - PLANET_RADIUS; // Roughly
        const v = pPos.clone().normalize();
        const lat = Math.abs(v.y);
        // Moisture? Need noise function here or just random check?
        // Accessing original noise is hard without keeping instance.
        // We'll just show "Analyzing..." or basic info.
        // Let's use height/lat.
        const normH = height / 150;
        getBiomeData(normH, lat, 0.5); // Default moisture
    }
    
    ocean.update(sec);
    atmosphere.update(sunDir);
    clouds.update(delta);
    aurora.update(sec, sunDir);
    celestials.update(sec);
    spaceDust.update(delta);
    
    renderer.render(scene, camera);
}

animate();
