import * as THREE from 'three';

const AtmosphereVertexShader = `
varying vec3 vNormal;
varying vec3 vPosition;
varying vec3 vViewPosition;

void main() {
  vNormal = normalize(normalMatrix * normal);
  vPosition = (modelMatrix * vec4(position, 1.0)).xyz;
  vec4 mvPosition = modelViewMatrix * vec4(position, 1.0);
  vViewPosition = -mvPosition.xyz;
  gl_Position = projectionMatrix * mvPosition;
}
`;

const AtmosphereFragmentShader = `
varying vec3 vNormal;
varying vec3 vPosition;
varying vec3 vViewPosition;

uniform vec3 uColor;
uniform vec3 uSunDirection;
uniform float uAtmosphereRadius;
uniform float uPlanetRadius;

void main() {
  vec3 viewDir = normalize(vViewPosition);
  vec3 normal = normalize(vNormal);
  
  // Fresnel / Limb
  float viewDot = dot(viewDir, normal);
  float edge = 1.0 - max(viewDot, 0.0);
  
  // Limb Glow (Sharper falloff at edge)
  float rim = pow(edge, 4.0);
  
  // Sun influence (day/night)
  float sunDot = dot(normal, uSunDirection);
  
  // Day/Night gradient
  float sunFactor = smoothstep(-0.3, 0.5, sunDot);
  
  // Backside scattering (sunset/sunrise glow)
  // If sun is behind planet relative to view? No, this is surface normal.
  // Just normal day/night scattering.
  
  // Color mixing
  vec3 color = uColor;
  
  // Add sunset colors at terminator?
  if (sunDot > -0.2 && sunDot < 0.2) {
      color = mix(vec3(1.0, 0.4, 0.1), uColor, smoothstep(-0.2, 0.2, sunDot));
  }
  
  // Base Haze (Rayleigh)
  // More blue at edges, but fades on night side
  float haze = pow(edge, 2.0) * 0.5;
  
  // Combine
  // Alpha depends on Rim and Sun
  float alpha = (rim * 0.8 + haze * 0.2) * sunFactor;
  
  // Add a faint glow on night side?
  alpha += rim * 0.1; 
  
  // Force alpha clamp
  alpha = clamp(alpha, 0.0, 1.0);
  
  gl_FragColor = vec4(color, alpha);
}
`;

export class PlanetAtmosphere {
  constructor(radius = 1000) {
    this.geometry = new THREE.SphereGeometry(radius * 1.05, 128, 64); // 5% higher
    
    this.material = new THREE.ShaderMaterial({
      vertexShader: AtmosphereVertexShader,
      fragmentShader: AtmosphereFragmentShader,
      uniforms: {
        uColor: { value: new THREE.Color(0x4488ff) },
        uSunDirection: { value: new THREE.Vector3(1, 0.5, 0).normalize() },
        uAtmosphereRadius: { value: radius * 1.2 },
        uPlanetRadius: { value: radius }
      },
      transparent: true,
      side: THREE.FrontSide, // Visible from outside
      blending: THREE.AdditiveBlending,
      depthWrite: false
    });

    this.mesh = new THREE.Mesh(this.geometry, this.material);
  }
  
  update(sunDir) {
      this.material.uniforms.uSunDirection.value.copy(sunDir);
  }
}
