import * as THREE from 'three';

const AuroraVertexShader = `
varying vec3 vPosition;
varying vec3 vNormal;
varying vec2 vUv;

void main() {
    vUv = uv;
    vNormal = normalize(normalMatrix * normal);
    vPosition = (modelMatrix * vec4(position, 1.0)).xyz;
    gl_Position = projectionMatrix * modelViewMatrix * vec4(position, 1.0);
}
`;

const AuroraFragmentShader = `
varying vec3 vPosition;
varying vec3 vNormal;
varying vec2 vUv;

uniform float uTime;
uniform vec3 uSunDirection;

// Simple noise (would usually import, but GLSL needs it inline)
// Simplex 3D noise function (simplified)
vec3 mod289(vec3 x) { return x - floor(x * (1.0 / 289.0)) * 289.0; }
vec4 mod289(vec4 x) { return x - floor(x * (1.0 / 289.0)) * 289.0; }
vec4 permute(vec4 x) { return mod289(((x*34.0)+1.0)*x); }
vec4 taylorInvSqrt(vec4 r) { return 1.79284291400159 - 0.85373472095314 * r; }
float snoise(vec3 v) {
  const vec2  C = vec2(1.0/6.0, 1.0/3.0) ;
  const vec4  D = vec4(0.0, 0.5, 1.0, 2.0);
  vec3 i  = floor(v + dot(v, C.yyy) );
  vec3 x0 = v - i + dot(i, C.xxx) ;
  vec3 g = step(x0.yzx, x0.xyz);
  vec3 l = 1.0 - g;
  vec3 i1 = min( g.xyz, l.zxy );
  vec3 i2 = max( g.xyz, l.zxy );
  vec3 x1 = x0 - i1 + C.xxx;
  vec3 x2 = x0 - i2 + C.yyy; // 2.0*C.x = 1/3 = C.y
  vec3 x3 = x0 - D.yyy;      // -1.0+3.0*C.x = -0.5 = -D.y
  i = mod289(i);
  vec4 p = permute( permute( permute(
             i.z + vec4(0.0, i1.z, i2.z, 1.0 ))
           + i.y + vec4(0.0, i1.y, i2.y, 1.0 ))
           + i.x + vec4(0.0, i1.x, i2.x, 1.0 ));
  float n_ = 0.142857142857; // 1.0/7.0
  vec3  ns = n_ * D.wyz - D.xzx;
  vec4 j = p - 49.0 * floor(p * ns.z * ns.z);  //  mod(p,7*7)
  vec4 x_ = floor(j * ns.z);
  vec4 y_ = floor(j - 7.0 * x_ );    // mod(j,N)
  vec4 x = x_ *ns.x + ns.yyyy;
  vec4 y = y_ *ns.x + ns.yyyy;
  vec4 h = 1.0 - abs(x) - abs(y);
  vec4 b0 = vec4( x.xy, y.xy );
  vec4 b1 = vec4( x.zw, y.zw );
  vec4 s0 = floor(b0)*2.0 + 1.0;
  vec4 s1 = floor(b1)*2.0 + 1.0;
  vec4 sh = -step(h, vec4(0.0));
  vec4 a0 = b0.xzyw + s0.xzyw*sh.xxyy ;
  vec4 a1 = b1.xzyw + s1.xzyw*sh.zzww ;
  vec3 p0 = vec3(a0.xy,h.x);
  vec3 p1 = vec3(a0.zw,h.y);
  vec3 p2 = vec3(a1.xy,h.z);
  vec3 p3 = vec3(a1.zw,h.w);
  vec4 norm = taylorInvSqrt(vec4(dot(p0,p0), dot(p1,p1), dot(p2, p2), dot(p3,p3)));
  p0 *= norm.x;
  p1 *= norm.y;
  p2 *= norm.z;
  p3 *= norm.w;
  return 105.0 * dot( p0, (x0.xyz - 0.0) ) + 1.0; // Scaled? snoise usually -1 to 1.
}
// End noise

void main() {
    vec3 normal = normalize(vNormal);
    
    // Latitude check (Poles)
    // Assuming y is Up.
    float lat = abs(normal.y);
    // Make it strictly high latitude (e.g. > 60 degrees)
    float latFactor = smoothstep(0.8, 0.95, lat);
    
    if (latFactor < 0.01) discard;
    
    // Night check
    float sunDot = dot(normal, uSunDirection);
    // Only on night side
    float nightFactor = smoothstep(0.0, -0.3, sunDot);
    
    if (nightFactor < 0.01) discard;
    
    // Noise
    float time = uTime * 0.2; // Slower
    // Move noise
    float n = snoise(vPosition * 0.003 + vec3(0.0, time, 0.0));
    float n2 = snoise(vPosition * 0.005 - vec3(0.0, time * 1.5, 0.0));
    
    // Combine noise for wispy look
    float combined = (n + n2) * 0.5;
    
    // Banding / curtain effect
    float band = smoothstep(0.2, 0.6, combined);
    
    // Color
    // Green/Teal base (0x00ff88 -> 0x00ffcc), Purple tips (0x8866ff)
    vec3 c1 = vec3(0.0, 1.0, 0.53); // Bright Green
    vec3 c2 = vec3(0.53, 0.4, 1.0); // Soft Purple
    
    vec3 col = mix(c1, c2, smoothstep(0.3, 0.8, n));
    
    // Fade out at edges of bands
    float alpha = band * latFactor * nightFactor * 0.25; // More subtle (was 0.4)
    
    gl_FragColor = vec4(col, alpha);
}
`;

export class PlanetAurora {
    constructor(radius = 1000) {
        this.geometry = new THREE.SphereGeometry(radius * 1.08, 64, 64); // High up
        this.material = new THREE.ShaderMaterial({
            vertexShader: AuroraVertexShader,
            fragmentShader: AuroraFragmentShader,
            uniforms: {
                uTime: { value: 0 },
                uSunDirection: { value: new THREE.Vector3(1,0,0) }
            },
            transparent: true,
            side: THREE.DoubleSide,
            blending: THREE.AdditiveBlending,
            depthWrite: false
        });
        this.mesh = new THREE.Mesh(this.geometry, this.material);
    }
    
    update(time, sunDir) {
        this.material.uniforms.uTime.value = time;
        this.material.uniforms.uSunDirection.value.copy(sunDir);
    }
}
