import * as THREE from 'three';
import { createNoise3D } from 'simplex-noise';

export class PlanetClouds {
  constructor(radius = 1000) {
    // Increased from 1.02 to 1.05 to reduce z-fighting with terrain at distance
    this.geometry = new THREE.SphereGeometry(radius * 1.05, 128, 64); // 5% height
    
    // Generate cloud texture (reduced for performance)
    const size = 2048; // Reduced from 2048 for faster startup
    const data = new Uint8Array(size * size * 4);
    const noise3D = createNoise3D();

    // FBM Helper
    const fbm = (x, y, z) => {
        let total = 0;
        let amplitude = 1;
        let frequency = 1;
        let maxVal = 0;
        for (let i = 0; i < 4; i++) { // 4 Octaves (reduced from 6)
            total += noise3D(x * frequency, y * frequency, z * frequency) * amplitude;
            maxVal += amplitude;
            amplitude *= 0.5;
            frequency *= 2;
        }
        return (total / maxVal) * 0.5 + 0.5;
    };
    
    for (let i = 0; i < size; i++) {
        for (let j = 0; j < size; j++) {
            // Spherical mapping... approximate with 3D noise on sphere surface
            // Map UV to sphere coordinates
            const u = i / size;
            const v = j / size;
            
            const theta = u * Math.PI * 2;
            const phi = v * Math.PI;
            
            const x = Math.sin(phi) * Math.cos(theta);
            const y = Math.cos(phi);
            const z = Math.sin(phi) * Math.sin(theta);
            
            const scale = 1.5; // Base cloud scale
            const n = fbm(x * scale, y * scale, z * scale);
            
            // Cloud threshold
            let val = 0;
            const threshold = 0.45;
            if (n > threshold) {
                // Soft edge
                val = (n - threshold) / (1.0 - threshold);
                val = val * val * (3.0 - 2.0 * val); // Smoothstep-like
            }
            
            // Opacity curve
            const alpha = Math.floor(val * 255);
            
            const index = (i + j * size) * 4;
            data[index] = 255;     // R
            data[index + 1] = 255; // G
            data[index + 2] = 255; // B
            data[index + 3] = alpha; // A
        }
    }
    
    const texture = new THREE.DataTexture(data, size, size, THREE.RGBAFormat);
    texture.needsUpdate = true;
    texture.wrapS = THREE.RepeatWrapping;
    texture.wrapT = THREE.RepeatWrapping; 
    texture.magFilter = THREE.LinearFilter;
    texture.minFilter = THREE.LinearMipMapLinearFilter;
    texture.generateMipmaps = true;

    this.material = new THREE.MeshStandardMaterial({
      map: texture,
      transparent: true,
      opacity: 0.9,
      side: THREE.DoubleSide,
      alphaTest: 0.05, // Low alpha test to keep soft edges but avoid z-sorting issues if possible? 
      // Standard material with transparency needs sorting.
      // alphaTest turns it into cutout. If we want soft, we need depthWrite: false.
      depthWrite: false,
    });

    this.mesh = new THREE.Mesh(this.geometry, this.material);
    this.mesh.castShadow = true;
  }

  update(delta) {
    this.mesh.rotation.y += delta * 0.02; // Slower rotation
  }
}
