import * as THREE from 'three';
import { createNoise3D } from 'simplex-noise';

export class PlanetOcean {
  constructor(radius = 1000, terrain = null) {
    this.radius = radius;
    this.geometry = new THREE.SphereGeometry(radius, 128, 64);
    
    // Create a smooth noise texture for normals
    const size = 1028;
    const data = new Uint8Array(size * size * 4);
    const noise3D = createNoise3D();
    
    const scale = 5.0; // Noise scale
    
    for (let i = 0; i < size; i++) {
        for (let j = 0; j < size; j++) {
            const u = i / size;
            const v = j / size;
            
            // Map to sphere surface approx or just tileable 3D noise
            // Using 3D noise with time=0
            const x = Math.cos(u * Math.PI * 2) * scale;
            const y = Math.sin(u * Math.PI * 2) * scale;
            const z = v * scale;
            
            const n = noise3D(x, y, z); // -1 to 1
            
            // Normalize to 0-255
            const val = Math.floor((n * 0.5 + 0.5) * 255);
            
            const index = (i + j * size) * 4;
            data[index] = val;     // R
            data[index + 1] = val; // G
            data[index + 2] = 255; // B (Normal map Z usually 1)
            data[index + 3] = 255; // A
        }
    }
    
    const noiseTexture = new THREE.DataTexture(data, size, size, THREE.RGBAFormat);
    noiseTexture.needsUpdate = true;
    noiseTexture.wrapS = THREE.RepeatWrapping;
    noiseTexture.wrapT = THREE.RepeatWrapping;

    this.material = new THREE.MeshPhysicalMaterial({
      color: 0x0066aa, // Slightly lighter blue
      metalness: 0.1,
      roughness: 0.2, // Smoother
      transmission: 0.8, // More transparent
      thickness: 20.0, // Depth for refraction
      clearcoat: 1.0,
      clearcoatRoughness: 0.1,
      normalMap: noiseTexture,
      normalScale: new THREE.Vector2(0.3, 0.3), // Gentler waves
      // Polygon offset to reduce z-fighting with terrain at shorelines
      polygonOffset: true,
      polygonOffsetFactor: 1,
      polygonOffsetUnits: 1
    });
    
    this.noiseTexture = noiseTexture;
    this.mesh = new THREE.Mesh(this.geometry, this.material);

    if (terrain) {
        this.createFoam(terrain);
    }
  }

  createFoam(terrain) {
    const width = 512;
    const height = 256;
    const canvas = document.createElement('canvas');
    canvas.width = width;
    canvas.height = height;
    const ctx = canvas.getContext('2d');
    
    const imgData = ctx.createImageData(width, height);
    const data = imgData.data;
    
    const vec = new THREE.Vector3();

    for (let y = 0; y < height; y++) {
        for (let x = 0; x < width; x++) {
            const u = x / width;
            const v = 1 - (y / height); // Flip Y for UV
            
            // Spherical coordinates
            // phi: 0 (North) to PI (South)
            // theta: 0 to 2PI
            const phi = (1 - v) * Math.PI; 
            const theta = u * 2 * Math.PI;
            
            vec.setFromSphericalCoords(this.radius, phi, theta);
            
            const surfaceH = terrain.getSurfaceHeight(vec.x, vec.y, vec.z);
            const depth = this.radius - surfaceH;
            
            // Foam logic
            // Shallow water: depth > 0 (underwater) and depth < small_amount
            // Also allow slightly negative depth (just on land) to blend? No, keep it water.
            
            let alpha = 0;
            const foamDepth = 20; // Depth where foam disappears
            
            if (depth > -2 && depth < foamDepth) { // Allow slight overlap with land to prevent gaps
                // Normalized foam intensity
                // Closer to 0 depth (shore) -> 1
                // Closer to foamDepth -> 0
                
                // Remap depth from [-2, 20] to [1, 0] approx
                let d = depth;
                if (d < 0) d = 0;
                
                const ratio = 1 - (d / foamDepth);
                
                // Non-linear falloff
                alpha = Math.pow(ratio, 3) * 200; // Max alpha 200
                
                // Add some noise to the foam edge? 
                // We can use simple random here or re-use coordinate noise if we want.
                // For simplicity, just gradient.
                if (Math.random() > 0.8 * ratio) {
                   alpha *= 0.5; // Break it up a bit
                }
            }
            
            const index = (x + y * width) * 4;
            data[index] = 255;     // R
            data[index + 1] = 255; // G
            data[index + 2] = 255; // B
            data[index + 3] = Math.floor(alpha); // A
        }
    }
    
    ctx.putImageData(imgData, 0, 0);
    const foamTexture = new THREE.CanvasTexture(canvas);
    
    const foamGeo = new THREE.SphereGeometry(this.radius + 0.8, 128, 64); // Just above water
    const foamMat = new THREE.MeshBasicMaterial({
        map: foamTexture,
        transparent: true,
        color: 0xffffff,
        depthWrite: false,
        side: THREE.DoubleSide
    });
    
    this.foamMesh = new THREE.Mesh(foamGeo, foamMat);
    this.mesh.add(this.foamMesh);
  }

  update(time) {
    if (this.material.normalMap) {
        // Animate texture offset for wave movement
        this.material.normalMap.offset.x = (time * 0.02) % 1;
        this.material.normalMap.offset.y = (time * 0.01) % 1;
    }
  }
}
