import * as THREE from 'three';
import { fbm } from '../utils/noise.js';
import { getBiomeData } from './biomes.js';

export class PlanetTerrain {
  constructor(radius = 1000) {
    this.radius = radius;
    // SphereGeometry(radius, 128, 64) -> ~8k vertices for good performance
    const terrain_detail = 1000
    this.geometry = new THREE.SphereGeometry(radius, terrain_detail, terrain_detail / 2);
    
    this.material = new THREE.MeshStandardMaterial({
      vertexColors: true,
      roughness: 0.8,
      metalness: 0.1,
      flatShading: false
    });

    this.mesh = new THREE.Mesh(this.geometry, this.material);
    this.mesh.castShadow = true;
    this.mesh.receiveShadow = true;
    
    this.generateTerrain();
  }

  generateTerrain() {
    const posAttribute = this.geometry.attributes.position;
    const count = posAttribute.count;
    const colors = [];
    
    const noiseScale = 0.002;
    
    // Helper to get height from noise
    this.getHeight = (x, y, z) => {
      // Base continent noise
      const n1 = fbm(x, y, z, { scale: 0.0005, octaves: 4, persistence: 0.5 }); 
      
      // Mountain noise
      const n2 = fbm(x, y, z, { scale: 0.003, octaves: 6, persistence: 0.5 });
      
      // Continents
      let baseH = n1;
      
      // Normalized height (-1 to 1 relative to "surface")
      let normH = 0;
      
      if (baseH < 0.45) {
          // Ocean floor
          normH = -0.2 - (0.45 - baseH) * 0.5; 
      } else {
          // Land
          normH = 0.0 + (baseH - 0.45) * 0.5; // Base land height
          
          // Add mountains
          normH += n2 * 0.15 * (baseH); // More mountains in centers of continents
      }
      
      const heightScale = 150; 
      
      return normH * heightScale;
    };

    // For moisture noise
    const moistureOffset = 1000; 

    for (let i = 0; i < count; i++) {
      const x = posAttribute.getX(i);
      const y = posAttribute.getY(i);
      const z = posAttribute.getZ(i);

      // Normalize to get direction
      const v = new THREE.Vector3(x, y, z).normalize();
      
      // Calculate Height
      const h = this.getHeight(x, y, z);
      
      // New Position
      const newPos = v.clone().multiplyScalar(this.radius + h);
      
      posAttribute.setXYZ(i, newPos.x, newPos.y, newPos.z);

      // Biome Info
      // Normalized Height for biome: 0 is sea level.
      const normH = h / 150; // Approximate max height
      
      // Latitude
      const lat = Math.abs(v.y); // 0 equator, 1 pole
      
      // Moisture
      const m = fbm(x + moistureOffset, y, z + moistureOffset, { scale: 0.001, octaves: 2 });
      
      const biome = getBiomeData(normH, lat, m);
      
      colors.push(biome.color.r, biome.color.g, biome.color.b);
    }

    this.geometry.setAttribute('color', new THREE.Float32BufferAttribute(colors, 3));
    this.geometry.computeVertexNormals();
  }

  // Helper for physics to get surface height at arbitrary pos
  getSurfaceHeight(x, y, z) {
      // Normalize input to get direction on unit sphere, then scale
      const v = new THREE.Vector3(x, y, z).normalize();
      // We need the original coordinates on the sphere surface (radius) to sample noise
      const sx = v.x * this.radius;
      const sy = v.y * this.radius;
      const sz = v.z * this.radius;
      
      const h = this.getHeight(sx, sy, sz);
      return this.radius + h;
  }
}
