import * as THREE from 'three';
import { fbm } from '../utils/noise.js';
import { getBiomeData } from './biomes.js';

export class PlanetVegetation {
  constructor(radius, terrain) {
    this.radius = radius;
    this.terrain = terrain;
    this.group = new THREE.Group();
    
    this.generateVegetation();
  }

  generateVegetation() {
    const count = 10000; // Reduced for performance (was 2000)
    const dummy = new THREE.Object3D();
    
    // Tree Geometries
    // Trunk
    const trunkGeo = new THREE.CylinderGeometry(2, 3, 10, 5);
    trunkGeo.translate(0, 5, 0); // Pivot at bottom
    const trunkMat = new THREE.MeshStandardMaterial({ color: 0x8B4513, roughness: 0.9 });
    
    // Foliage (Cone)
    const foliageGeo = new THREE.ConeGeometry(8, 20, 5);
    foliageGeo.translate(0, 20, 0); // Sit on top of trunk (10 high) + half height (10) = 20
    const foliageMat = new THREE.MeshStandardMaterial({ color: 0x2d5a27, roughness: 0.8 });

    // Instanced Meshes
    const trunkMesh = new THREE.InstancedMesh(trunkGeo, trunkMat, count);
    const foliageMesh = new THREE.InstancedMesh(foliageGeo, foliageMat, count);
    
    trunkMesh.receiveShadow = true;
    trunkMesh.castShadow = true;
    foliageMesh.receiveShadow = true;
    foliageMesh.castShadow = true;

    let instanceIndex = 0;
    const moistureOffset = 1000; 

    // Try to place trees
    // We sample more points than count to find valid biome spots
    const samples = 50000;
    
    for (let i = 0; i < samples; i++) {
        if (instanceIndex >= count) break;

        // Random point on sphere
        const u = Math.random();
        const v = Math.random();
        const theta = 2 * Math.PI * u;
        const phi = Math.acos(2 * v - 1);
        
        const x = this.radius * Math.sin(phi) * Math.cos(theta);
        const y = this.radius * Math.sin(phi) * Math.sin(theta);
        const z = this.radius * Math.cos(phi);
        
        // Get Terrain Info
        // Note: terrain.getHeight expects coordinates on the base sphere surface
        // which x,y,z are.
        let h = 0;
        try {
            h = this.terrain.getHeight(x, y, z);
        } catch (e) {
            console.warn("Terrain height function not available yet");
            break;
        }

        // Biome Check
        const normH = h / 150;
        // Latitude (y is up in 3D? Wait. In main.js camera is 0,0,Radius*3. 
        // In terrain.js: const lat = Math.abs(v.y); 
        // Standard Three.js Y is up. 
        // My random sphere gen: x,y,z.
        // Let's align coords.
        const dir = new THREE.Vector3(x, y, z).normalize();
        const lat = Math.abs(dir.y);
        
        const m = fbm(x + moistureOffset, y, z + moistureOffset, { scale: 0.001, octaves: 2 });
        
        const biome = getBiomeData(normH, lat, m);
        
        if (biome.name === 'Grassland' || biome.name === 'Forest') {
            // Place Tree
            
            // Position
            const pos = dir.clone().multiplyScalar(this.radius + h);
            dummy.position.copy(pos);
            
            // Rotation: Align +Y with normal (dir)
            // Default cylinder points +Y.
            // Quaternion from +Y to dir.
            const up = new THREE.Vector3(0, 1, 0);
            dummy.quaternion.setFromUnitVectors(up, dir);
            
            // Scale variation
            const s = 1.0 + Math.random() * 0.5;
            dummy.scale.set(s, s, s);
            
            dummy.updateMatrix();
            
            trunkMesh.setMatrixAt(instanceIndex, dummy.matrix);
            foliageMesh.setMatrixAt(instanceIndex, dummy.matrix);
            
            instanceIndex++;
        }
    }
    
    trunkMesh.count = instanceIndex;
    foliageMesh.count = instanceIndex;
    
    this.group.add(trunkMesh);
    this.group.add(foliageMesh);
    
    console.log(`Generated ${instanceIndex} trees.`);
  }
}
