import * as THREE from 'three';

export class PlanetRings {
    constructor(planetRadius) {
        const innerRadius = planetRadius * 1.4;
        const outerRadius = planetRadius * 2.3;
        
        // Ring Geometry: (innerRadius, outerRadius, thetaSegments, phiSegments, thetaStart, thetaLength)
        // High segment count for roundness
        const geometry = new THREE.RingGeometry(innerRadius, outerRadius, 128);
        
        // UV Mapping correction for RingGeometry to map texture radially?
        // Default RingGeometry UVs map:
        // Inner ring is (u, 0) ?? Actually RingGeometry UVs are planar projection by default in older Three.js, 
        // but in newer ones they might be polar.
        // Let's check standard behavior: RingGeometry usually maps the texture linearly along the ring if we want bands?
        // Actually, standard RingGeometry UVs are planar (projected on XY). 
        // This makes applying a radial gradient texture difficult without custom UVs.
        
        // Let's fix UVs to be linear: U = angle, V = radius
        const posAttribute = geometry.attributes.position;
        const uvAttribute = geometry.attributes.uv;
        
        const center = new THREE.Vector3();
        const vec = new THREE.Vector3();
        
        for (let i = 0; i < posAttribute.count; i++) {
            vec.fromBufferAttribute(posAttribute, i);
            const radius = vec.length();
            
            // Normalize radius between inner (0) and outer (1) for V
            const v = (radius - innerRadius) / (outerRadius - innerRadius);
            
            // Only using V for the bands (radial texture)
            // We can ignore U for concentric rings, or use it for azimuthal variation
            uvAttribute.setXY(i, 0, v); // We only care about V for the gradient texture
        }
        
        const texture = this.createRingTexture();
        
        const material = new THREE.MeshStandardMaterial({
            map: texture,
            transparent: true,
            opacity: 0.9,
            side: THREE.DoubleSide,
            color: 0xffffff,
            roughness: 0.8,
            metalness: 0.1
        });
        
        this.mesh = new THREE.Mesh(geometry, material);
        this.mesh.castShadow = true;
        this.mesh.receiveShadow = true;
        
        // Orientation
        // RingGeometry creates on XY plane facing Z.
        // Rotate -90 deg on X to lay flat on XZ plane (normal pointing up Y).
        this.mesh.rotation.x = -Math.PI / 2;
        
        // Add planetary tilt (approx 25 degrees)
        this.mesh.rotation.x += THREE.MathUtils.degToRad(25);
        this.mesh.rotation.y = THREE.MathUtils.degToRad(15); // Slight twist
    }
    
    createRingTexture() {
        const width = 512;
        const height = 64; // Slight height to handle mipmapping better? Or just 1.
        
        const canvas = document.createElement('canvas');
        canvas.width = width;
        canvas.height = height;
        const ctx = canvas.getContext('2d');
        
        // Fill background transparent
        ctx.fillStyle = 'rgba(0,0,0,0)';
        ctx.fillRect(0, 0, width, height);
        
        // Since we mapped UVs such that V (vertical in texture) corresponds to Radius.
        // We want the gradient to go from bottom (inner) to top (outer) or vice versa.
        // Let's draw horizontal lines (or vertical? wait).
        // UV setXY(i, 0, v). V varies from 0 to 1.
        // In Canvas, Y=0 is top. V=0 is usually bottom in GL.
        // So bands should be horizontal across the texture?
        // Wait. If UV is (0, v), we are sampling the texture at U=0, V=variable.
        // So we need the gradient to vary along the V axis (Y axis of canvas).
        
        const gradient = ctx.createLinearGradient(0, height, 0, 0); // From bottom (V=0) to top (V=1)
        
        // Colors:
        // Transparencies (gaps)
        // Dust colors: #CDBA96 (DarkTan), #F5DEB3 (Wheat), #8B7D6B (Bisque4)
        
        const bands = [
            { pos: 0.00, color: 'rgba(0,0,0,0)' }, // Inner gap
            { pos: 0.10, color: 'rgba(139, 125, 107, 0.4)' },
            { pos: 0.15, color: 'rgba(205, 186, 150, 0.8)' },
            { pos: 0.25, color: 'rgba(100, 90, 80, 0.1)' }, // Gap (Cassini division-ish)
            { pos: 0.28, color: 'rgba(200, 180, 140, 0.7)' },
            { pos: 0.40, color: 'rgba(245, 222, 179, 0.9)' }, // Bright main ring
            { pos: 0.60, color: 'rgba(210, 190, 160, 0.8)' },
            { pos: 0.65, color: 'rgba(100, 90, 80, 0.2)' }, // Gap
            { pos: 0.70, color: 'rgba(180, 160, 130, 0.5)' },
            { pos: 0.90, color: 'rgba(120, 110, 100, 0.3)' },
            { pos: 1.00, color: 'rgba(0,0,0,0)' }  // Fade out
        ];
        
        bands.forEach(band => {
            gradient.addColorStop(band.pos, band.color);
        });
        
        ctx.fillStyle = gradient;
        ctx.fillRect(0, 0, width, height);
        
        // Add some noise/grain
        const imageData = ctx.getImageData(0,0, width, height);
        const data = imageData.data;
        for(let i=0; i < data.length; i+=4) {
            if (data[i+3] > 0) { // If not fully transparent
                const noise = (Math.random() - 0.5) * 20;
                data[i] = Math.max(0, Math.min(255, data[i] + noise));
                data[i+1] = Math.max(0, Math.min(255, data[i+1] + noise));
                data[i+2] = Math.max(0, Math.min(255, data[i+2] + noise));
            }
        }
        ctx.putImageData(imageData, 0, 0);
        
        const tex = new THREE.CanvasTexture(canvas);
        tex.wrapS = THREE.RepeatWrapping;
        tex.wrapT = THREE.ClampToEdgeWrapping; // Don't repeat radially
        //tex.magFilter = THREE.NearestFilter; // Maybe Linear is better for rings
        tex.minFilter = THREE.LinearMipMapLinearFilter;
        
        return tex;
    }
}
